#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>

#include "scalejpg.h"
#include "dirent.h"

#ifdef WIN32
#define PATH_SEPARATOR "\\"
#define alloca _alloca
#define os_mkdir(d) mkdir(d)
#else
#define PATH_SEPARATOR "/"
#define os_mkdir(d) mkdir(d, 0755)
#endif

static int is_extension(const char *fname, const char *exts)
{
	const char *e;
	const char *q;
	char *s; 

	/* remove ? part */
	q = strchr(fname, '?');
	if (q)
	{
		s = alloca(q - fname + 1);
		memcpy(s, fname, q - fname);
		s[q - fname] = '\0';
		fname = s;
	}

	e = strrchr(fname, '.');
	if (e)
	{
		while(*exts)
		{
			if (!stricmp(exts, e))
				return TRUE;
			exts += strlen(exts) + 1;
		}
	}
	return FALSE;
}

static char *basename(const char *path)
{
	const char *result;

	result = path;
	while(*path)
	{
		if ((*path == '\\') || (*path == '/'))
			result = path + 1;
		path++;
	}
	return (char *) result;
}

static void my_stat(const char *path, struct stat *s)
{
	int err;
	err = stat(path, s);
	if (err)
	{
		fprintf(stderr, "Failed to stat() '%s'\r\n", path);
		exit(-1);
	}
}

static void copy_file(const char *dest, const char *source)
{
	size_t sz;
	FILE *destf;
	FILE *sourcef;
	char buf[4096];
	
	destf = fopen(dest, "wb");
	sourcef = fopen(source, "rb");
	while(!feof(sourcef))
	{
		sz = fread(buf, 1, sizeof(buf), sourcef);
		fwrite(buf, 1, sz, destf);
	}
	fclose(sourcef);
	fclose(destf);
}

static void unescape(char *dest, const char *source)
{
	char buf[3];
	char c;
	int val;
	int i;

	for (i = 0; source[i]; i++)
	{
		c = source[i];
		if ((c == '%') && isxdigit(source[i+1]) && isxdigit(source[i+2]))
		{
			memcpy(buf, source + i + 1, 2);
			buf[2] = '\0';
			sscanf(buf, "%x", &val);
			c = (int) val;
			source += 2;
		}
		*(dest++) = c;
	}
	*dest = '\0';
}

static void process_jpeg(char *jpegname, const char *dest_dirname, const char *source_dirname)
{
	double scale;
	char *s2;
	char *s3;
	char *dest_jpeg;
	char *source_jpeg;
	struct stat s;

	s2 = strrchr(jpegname, '?');
	if (s2)
	{
		*(s2++) = '\0';
		scale = atof(s2);
	}
	else
	{
		scale = 0;
	}
	source_jpeg = alloca(strlen(source_dirname) + strlen(jpegname) + 2);
	strcpy(source_jpeg, source_dirname);
	strcat(source_jpeg, PATH_SEPARATOR);
	unescape(source_jpeg + strlen(source_jpeg), jpegname);

	if (s2)
	{
		s3 = strrchr(jpegname, '.');
		sprintf(s3, "_%g.jpg", scale);
		s3 = strchr(s3, '.');
		if (isdigit(s3[1]))
			*s3 = '_';
	}

	dest_jpeg = alloca(strlen(dest_dirname) + strlen(jpegname) + 2);
	strcpy(dest_jpeg, dest_dirname);
	strcat(dest_jpeg, PATH_SEPARATOR);
	unescape(dest_jpeg + strlen(dest_jpeg), jpegname);

	if (!stat(source_jpeg, &s))
	{
		if (stat(dest_jpeg, &s))
		{
			if (s2)
			{
				fprintf(stdout, "Scaling '%s'...\n", source_jpeg);
				scale_image(dest_jpeg, source_jpeg, scale);
			}
			else
			{
				copy_file(dest_jpeg, source_jpeg);
			}
		}
		else
		{
			fprintf(stdout, "Skipping '%s'...\n", source_jpeg);
		}
	}
	else
	{
		fprintf(stderr, "WARNING: Cannot find jpeg '%s'\n", source_jpeg);
	}
}

static void copy_html(const char *dest, const char *source)
{
	size_t sz;
	FILE *destf;
	FILE *sourcef;
	int state = 0;
	size_t i;
	size_t jpeg_filename_pos;
	char jpeg_filename[256];
	char buf[4096];
	char *dest_dirname;
	char *source_dirname;
	char *s1;
	char *s2;
	
	dest_dirname = alloca(strlen(dest) + 1);
	strcpy(dest_dirname, dest);
	*basename(dest_dirname) = '\0';
	source_dirname = alloca(strlen(source) + 1);
	strcpy(source_dirname, source);
	*basename(source_dirname) = '\0';

	destf = fopen(dest, "wb");
	sourcef = fopen(source, "rb");

	while(!feof(sourcef))
	{
		sz = fread(buf, 1, sizeof(buf), sourcef);
		for (i = 0; i < sz; i++)
		{
			switch(state) {
			case 0:
				if (buf[i] == '<')
				{
					state = 1;
				}
				fputc(buf[i], destf);
				break;
			
			case 1:
				if (buf[i] == '>')
				{
					state = 0;
				}
				else if (buf[i] == '=')
				{
					state = 2;
					jpeg_filename_pos = 0;
				}
				fputc(buf[i], destf);
				break;

			case 2:
				/* inside attribute */
				if (isspace(buf[i]) || (buf[i] == '>'))
				{
					/* remove quotes from filename */
					jpeg_filename[jpeg_filename_pos] = '\0';
					for (s1 = jpeg_filename; (*s1 == '\'') || (*s1 == '\"'); s1++)
						;
					for (s2 = s1+strlen(s1)-1; (*s2 == '\'') || (*s2 == '\"'); s2--)
						;
					s2[1] = '\0';
					if (is_extension(s1, ".jpg\0"))
					{
						/* we have a JPEG */
						process_jpeg(s1, dest_dirname, source_dirname);
					}

					/* output attribute filename */
					fputc('\"', destf);
					fputs(s1, destf);
					fputc('\"', destf);
					fputc(buf[i], destf);
					state = (buf[i] == '>') ? 0 : 1;
				}
				else
				{
					jpeg_filename[jpeg_filename_pos++] = buf[i];
				}
			}
		}
	}
	fclose(sourcef);
	fclose(destf);
}

static int image_munge(const char *dest, const char *source)
{
	struct stat s;
	DIR *d;
	struct dirent *ent;
	char *newdest;
	char *newsource;
	const char *source_basename;
	int err;

	/* does destination directory exist? */
	if (stat(dest, &s))
	{
		/* if not, create it */
		if (os_mkdir(dest))
		{
			fprintf(stderr, "Failed to create directory '%s'\r\n", dest);
			return -1;
		}
	}

	/* figure out target name */
	source_basename = basename(source);
	newdest = alloca(strlen(dest) + strlen(source_basename) + 2);
	strcpy(newdest, dest);
	strcat(newdest, PATH_SEPARATOR);
	strcat(newdest, source_basename);

	/* what is the source? */
	my_stat(source, &s);
	if (s.st_mode & S_IFDIR)
	{	
		/* browse directory */
		d = opendir(source);
		while((ent = readdir(d)) != NULL)
		{
			if (strcmp(ent->d_name, ".") && strcmp(ent->d_name, ".."))
			{
				newsource = alloca(strlen(source) + strlen(ent->d_name) + 2);
				strcpy(newsource, source);
				strcat(newsource, PATH_SEPARATOR);
				strcat(newsource, ent->d_name);
				err = image_munge(newdest, newsource);
				if (err)
					return err;
			}
		}
		closedir(d);
	}
	else
	{
		/* this is a file */
		if (is_extension(source, ".htm\0.html\0"))
		{
			/* this is an HTML file */
			copy_html(newdest, source);
		}
		else if (is_extension(source, ".jpg\0"))
		{
			/* this is a JPEG; ignore under the belief that it will have already been created when copying html files */
		}
		else
		{
			/* this is a normal file */
			copy_file(newdest, source);
		}
	}
	return 0;
}

int main(int argc, char *argv[])
{
	if (argc != 3)
	{
		fprintf(stderr, "%s: Usage\r\n", argv[0]);
		fprintf(stderr, "%s [source] [dest]\r\n", argv[0]);
		return -1;
	}
	return image_munge(argv[2], argv[1]);
}
